<?php

namespace Database\Factories;

use App\Models\AppBuilder;
use App\Services\PluginManager;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\AppBuilder>
 */
class AppBuilderFactory extends Factory
{
    protected $model = AppBuilder::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        // Default to first active platform, or fallback to android-webview
        $pluginManager = app(PluginManager::class);
        $firstPlatform = $pluginManager->getActivePlatforms()->first();
        $defaultPlatform = $firstPlatform ? $firstPlatform->getPlatformId() : 'android-webview';

        return [
            'name' => fake()->words(2, true).' Builder',
            'server_key' => Str::random(32),
            'url' => 'http://localhost',
            'port' => fake()->numberBetween(8000, 9000),
            'platforms' => [$defaultPlatform],
            'status' => 'active',
            'max_queue' => 5,
            'current_builds_count' => 0,
            'total_builds_processed' => 0,
            'is_default' => false,
        ];
    }

    /**
     * Indicate that the server is inactive.
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'inactive',
        ]);
    }

    /**
     * Indicate that the server handles AndroidWebView builds.
     */
    public function androidWebview(): static
    {
        return $this->state(fn (array $attributes) => [
            'platforms' => ['android-webview'],
        ]);
    }

    /**
     * Indicate that the server handles all active platforms.
     */
    public function allPlatforms(): static
    {
        return $this->state(function (array $attributes) {
            $pluginManager = app(PluginManager::class);

            return [
                'platforms' => $pluginManager->getActivePlatforms()
                    ->map(fn ($p) => $p->getPlatformId())
                    ->values()
                    ->toArray(),
            ];
        });
    }

    /**
     * Indicate that the server handles a specific platform.
     */
    public function forPlatform(string $platform): static
    {
        return $this->state(fn (array $attributes) => [
            'platforms' => [$platform],
        ]);
    }

    /**
     * Indicate that this is the default builder (TitanSystems hosted service).
     */
    public function default(): static
    {
        return $this->state(function (array $attributes) {
            $pluginManager = app(PluginManager::class);

            return [
                'name' => 'Default Builder',
                'url' => AppBuilder::DEFAULT_BUILDER_URL,
                'port' => AppBuilder::DEFAULT_BUILDER_PORT,
                'server_key' => 'placeholder', // Actual key comes from purchase_code via accessor
                'is_default' => true,
                'status' => 'active',
                'credit_cost' => 1,
                'platforms' => $pluginManager->getActivePlatforms()
                    ->map(fn ($p) => $p->getPlatformId())
                    ->values()
                    ->toArray(),
            ];
        });
    }
}
